//
//  ContentView.swift
//  VecoreSDKDemo
//
//  Created by Ramon Swilem on 16/09/2019.
//  Copyright © 2019 Ramon Swilem. All rights reserved.
//

import SwiftUI
import Combine
import VecoreSDKFramework

struct ContentView: View {
    @State private var manageKey: String = "11a11a0aa1ce4d1a7f576c3faf77b6eb70"
    @State private var connectionState: String = "Disconnected"
    @State private var status: String = ""
    @State private var isConnected: Bool = false
    private let feedbackGenerator: UISelectionFeedbackGenerator = UISelectionFeedbackGenerator()
    
    var body: some View {
        KeyboardHost {
            VStack {
                Header()
                
                VStack(alignment: .leading) {
                    Text("Manage Key")
                        .font(.vecoreSubheading)
                        .foregroundColor(.vecoreYellow)
                    TextField("Required", text: $manageKey)
                        .padding(.bottom, 15)
                }
                
                Spacer()
                
                if status.count > 0
                {
                    Text(status)
                        .foregroundColor(Color.vecoreGray)
                        .animation(.none)
                        .font(.vecoreRegular)
                    Divider()
                        .padding(.bottom)
                }
                
                if isConnected
                {
                    Button(action: disconnectVehicle) {
                        Text("Disconnect")
                            .modifier(PrimaryButton())
                    }
                    .padding(.bottom, 15)
                    
                    Button(action: {
                        self.sendVehicleCommand(VecoreVehicleCommand.Unlock)
                    }) {
                        Text("Open")
                            .modifier(AlternateButton())
                    }
                    .padding(.bottom, 15)
                    
                    Button(action: {
                        self.sendVehicleCommand(VecoreVehicleCommand.Lock)
                    }) {
                        Text("Close")
                            .modifier(AlternateButton())
                    }
                }
                else
                {
                    Button(action: searchVehicle) {
                        Text("Start search")
                            .modifier(PrimaryButton())
                    }
                }
                Text(connectionState)
                    .foregroundColor(Color.vecoreGray)
                    .animation(.none)
                    .font(.vecoreRegular)
                    .padding(.top, 15)
            }
            .animation(.linear(duration: 0.3))
            .padding()
        }
        .onAppear(perform: viewDidAppear)
    }
    
    private func viewDidAppear() {
        feedbackGenerator.prepare()
        
        // Set the debug handler so SDK messages are printed in the log and UI.
        VecoreSDK.sharedInstance.setDebugHandler { (logMessage) in
            print("SDK: \(logMessage)")
            self.status = logMessage
        }
        
        // Connect to the debugging environment
        VecoreSDK.sharedInstance.setDebugEnvironmentEnabled(true)
        
        // Authorize the SDK using a token provided by Vecore.
        VecoreSDK.sharedInstance.authorize(token: "YOUR-TOKEN")
    }
    
    private func searchVehicle() {
        feedbackGenerator.selectionChanged()
        
        guard self.manageKey.count > 0 else {
            self.status = "Please input a valid manage key. You'll find this value by fetching a reservation through the Vecore API."
            return
        }
        
        // Connect to the vehicle using the manage key. This manage key represents all information - connection type, reservation details and more.
        VecoreSDK.sharedInstance.connect(privateManageKey: self.manageKey) { (vehicleID: Int, connectionStatus: VecoreConnectionStatus) in
            print("Connection status callback -- Status code: \(connectionStatus.statusCode.rawValue) - Detail: \(connectionStatus.statusCodeDetail.rawValue) - Message: \(connectionStatus.message)")
            if connectionStatus.statusCode == .Unavailable {
                return
            }

            self.isConnected = (connectionStatus.statusCode == .Connected)
            switch connectionStatus.statusCode
            {
                case .Authenticating:
                    self.connectionState = "Authenticating..."
                    break
                
                case .Connected:
                    self.connectionState = "Connected"
                    break
                
                case .Connecting:
                    self.connectionState = "Connecting..."
                    break
                
                case .Disconnected:
                    self.connectionState = "Disconnected"
                    break
                
                case .Searching:
                    self.connectionState = "Searching..."
                    break
                
                case .Unavailable:
                    fallthrough
                default:
                    self.connectionState = "Unavailable"
                    break
            }
        }
    }
    
    private func disconnectVehicle() {
        feedbackGenerator.selectionChanged()
        
        guard self.manageKey.count > 0 else {
            self.status = "Could not disconnect. Invalid manage key specified"
            return
        }
        
        // Discard the connection for vehicle
        VecoreSDK.sharedInstance.disconnect(privateManageKey: self.manageKey)
    }
    
    private func sendVehicleCommand(_ command: VecoreVehicleCommand) {
        feedbackGenerator.selectionChanged()
        self.status = "Command sent."
        
        // Send a command, e.g. VecoreVehicleCommand.Lock
        VecoreSDK.sharedInstance.sendVehicleCommand(command)
    }
}

struct ContentView_Previews: PreviewProvider {
    static var previews: some View {
        ContentView()
    }
}
